<?php
/**
 * TuxoPay Webhook Handler
 */

defined('ABSPATH') || exit;

class TuxoPay_Webhook_Handler {

    /**
     * Handle webhook
     */
    public static function handle() {
        $raw_body = file_get_contents('php://input');
        $data = json_decode($raw_body, true);

        // Log the webhook
        self::log('Webhook received: ' . print_r($data, true));

        if (!$data || !isset($data['event'])) {
            self::log('Invalid webhook data');
            status_header(400);
            exit;
        }

        // Verify webhook signature (if implemented)
        // $signature = isset($_SERVER['HTTP_X_TUXOPAY_SIGNATURE']) ? $_SERVER['HTTP_X_TUXOPAY_SIGNATURE'] : '';
        // if (!self::verify_signature($raw_body, $signature)) {
        //     self::log('Invalid signature');
        //     status_header(401);
        //     exit;
        // }

        // Handle different event types
        switch ($data['event']) {
            case 'payment.status_changed':
                self::handle_payment_status_changed($data);
                break;
            default:
                self::log('Unknown event type: ' . $data['event']);
                break;
        }

        status_header(200);
        exit;
    }

    /**
     * Handle payment status changed event
     */
    private static function handle_payment_status_changed($data) {
        if (!isset($data['payment']) || !isset($data['metadata']['order_id'])) {
            self::log('Missing payment data or order ID');
            return;
        }

        $payment = $data['payment'];
        $order_id = $data['metadata']['order_id'];
        $payment_status = $payment['status'];
        $payment_id = $payment['id'];
        $payment_session_id = $payment['payment_session_id'];

        $order = wc_get_order($order_id);

        if (!$order) {
            self::log('Order not found: ' . $order_id);
            return;
        }

        // Verify payment session ID matches
        $stored_session_id = $order->get_meta('_tuxopay_payment_session_id');
        if ($stored_session_id && $stored_session_id !== $payment_session_id) {
            self::log('Payment session ID mismatch');
            return;
        }

        // Update payment ID if not already set
        if (!$order->get_meta('_tuxopay_payment_id')) {
            $order->update_meta_data('_tuxopay_payment_id', $payment_id);
        }

        // Store provider transaction ID if available
        if (isset($payment['provider_transaction_id']) && $payment['provider_transaction_id']) {
            $order->update_meta_data('_tuxopay_provider_transaction_id', $payment['provider_transaction_id']);
        }

        // Handle payment status
        switch ($payment_status) {
            case 'completed':
                // Check if already completed
                if ($order->has_status('completed') || $order->has_status('processing')) {
                    self::log('Order already processed: ' . $order_id);
                    return;
                }

                $order->payment_complete($payment_id);
                $order->add_order_note(
                    sprintf(
                        __('TuxoPay payment completed. Payment ID: %s', 'tuxopay-orchestrator'),
                        $payment_id
                    )
                );
                self::log('Payment completed for order: ' . $order_id);
                break;

            case 'failed':
                $error_message = isset($data['metadata']['error_message']) ? $data['metadata']['error_message'] : 'Payment failed';

                $order->update_status(
                    'failed',
                    sprintf(
                        __('TuxoPay payment failed. Payment ID: %s. Error: %s', 'tuxopay-orchestrator'),
                        $payment_id,
                        $error_message
                    )
                );
                self::log('Payment failed for order: ' . $order_id . ' - ' . $error_message);
                break;

            case 'pending':
                $order->update_status(
                    'on-hold',
                    sprintf(
                        __('TuxoPay payment pending. Payment ID: %s', 'tuxopay-orchestrator'),
                        $payment_id
                    )
                );
                self::log('Payment pending for order: ' . $order_id);
                break;

            default:
                self::log('Unknown payment status: ' . $payment_status);
                break;
        }

        $order->save();
    }

    /**
     * Verify webhook signature (optional - implement based on TuxoPay documentation)
     */
    private static function verify_signature($payload, $signature) {
        // Implement signature verification based on TuxoPay requirements
        // Example:
        // $secret = get_option('tuxopay_webhook_secret');
        // $calculated_signature = hash_hmac('sha256', $payload, $secret);
        // return hash_equals($calculated_signature, $signature);

        return true; // For now, accept all webhooks
    }

    /**
     * Log webhook activity
     */
    private static function log($message) {
        if (class_exists('WC_Logger')) {
            $logger = wc_get_logger();
            $logger->info($message, array('source' => 'tuxopay-webhook'));
        }
    }
}
