<?php
/**
 * TuxoPay Payment Gateway
 */

defined('ABSPATH') || exit;

class WC_Gateway_TuxoPay extends WC_Payment_Gateway {

    /**
     * Test mode enabled
     */
    public $testmode;

    /**
     * API Key
     */
    public $api_key;

    /**
     * Flow name
     */
    public $flow_name;

    /**
     * API URL
     */
    public $api_url;

    public function __construct() {
        $this->id                 = 'tuxopay';
        $this->icon               = '';
        $this->has_fields         = false;
        $this->method_title       = __('TuxoPay', 'tuxopay-orchestrator');
        $this->method_description = __('Accept payments via TuxoPay Orchestrator', 'tuxopay-orchestrator');
        $this->supports           = array(
            'products',
        );

        // Load settings
        $this->init_form_fields();
        $this->init_settings();

        // Get settings
        $this->title              = $this->get_option('title');
        $this->description        = $this->get_option('description');
        $this->enabled            = $this->get_option('enabled');
        $this->testmode           = 'yes' === $this->get_option('testmode');
        $this->api_key            = $this->testmode ? $this->get_option('test_api_key') : $this->get_option('live_api_key');
        $this->flow_name          = $this->testmode ? $this->get_option('test_flow_name') : $this->get_option('live_flow_name');
        $this->api_url            = $this->testmode ? 'https://orchestrator-staging.tuxopay.com' : 'https://orchestrator.tuxopay.com';

        // Hooks
        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
    }

    /**
     * Check if the gateway is available for use
     */
    public function is_available() {
        $is_available = parent::is_available();

        if (!$is_available) {
            return false;
        }

        // Check if API key is configured
        if (empty($this->api_key)) {
            return false;
        }

        // Check if flow name is configured
        if (empty($this->flow_name)) {
            return false;
        }

        return true;
    }

    /**
     * Initialize gateway settings form fields
     */
    public function init_form_fields() {
        $this->form_fields = array(
            'enabled' => array(
                'title'   => __('Enable/Disable', 'tuxopay-orchestrator'),
                'type'    => 'checkbox',
                'label'   => __('Enable TuxoPay', 'tuxopay-orchestrator'),
                'default' => 'no',
            ),
            'title' => array(
                'title'       => __('Title', 'tuxopay-orchestrator'),
                'type'        => 'text',
                'description' => __('This controls the title which the user sees during checkout.', 'tuxopay-orchestrator'),
                'default'     => __('Credit Card (TuxoPay)', 'tuxopay-orchestrator'),
                'desc_tip'    => true,
            ),
            'description' => array(
                'title'       => __('Description', 'tuxopay-orchestrator'),
                'type'        => 'textarea',
                'description' => __('This controls the description which the user sees during checkout.', 'tuxopay-orchestrator'),
                'default'     => __('Pay securely with your credit card via TuxoPay.', 'tuxopay-orchestrator'),
            ),
            'testmode' => array(
                'title'       => __('Test mode', 'tuxopay-orchestrator'),
                'label'       => __('Enable Test Mode', 'tuxopay-orchestrator'),
                'type'        => 'checkbox',
                'description' => __('Place the payment gateway in test mode using test API credentials.', 'tuxopay-orchestrator'),
                'default'     => 'yes',
                'desc_tip'    => true,
            ),
            'test_api_key' => array(
                'title'       => __('Test API Key', 'tuxopay-orchestrator'),
                'type'        => 'password',
                'description' => __('Get your API key from TuxoPay dashboard.', 'tuxopay-orchestrator'),
            ),
            'test_flow_name' => array(
                'title'       => __('Test Flow Name', 'tuxopay-orchestrator'),
                'type'        => 'text',
                'description' => __('The payment flow name configured in TuxoPay.', 'tuxopay-orchestrator'),
                'placeholder' => 'default_blackpay',
            ),
            'live_api_key' => array(
                'title'       => __('Live API Key', 'tuxopay-orchestrator'),
                'type'        => 'password',
                'description' => __('Get your API key from TuxoPay dashboard.', 'tuxopay-orchestrator'),
            ),
            'live_flow_name' => array(
                'title'       => __('Live Flow Name', 'tuxopay-orchestrator'),
                'type'        => 'text',
                'description' => __('The payment flow name configured in TuxoPay.', 'tuxopay-orchestrator'),
                'placeholder' => 'default_blackpay',
            ),
        );
    }

    /**
     * Process the payment
     */
    public function process_payment($order_id) {
        $order = wc_get_order($order_id);

        // Initiate payment flow
        $response = $this->initiate_payment_flow($order);

        if (is_wp_error($response)) {
            wc_add_notice(__('Payment error: ', 'tuxopay-orchestrator') . $response->get_error_message(), 'error');
            return array(
                'result' => 'failure',
                'messages' => $response->get_error_message(),
            );
        }

        if (!isset($response['success']) || !$response['success']) {
            $error_message = isset($response['error']) ? $response['error'] : 'Unknown error';
            wc_add_notice(__('Payment error: ', 'tuxopay-orchestrator') . $error_message, 'error');
            return array(
                'result' => 'failure',
                'messages' => $error_message,
            );
        }

        $payment_result = $response['data']['payment_result'];

        // Store payment session ID and payment ID
        if (isset($payment_result['payment_session_id'])) {
            $order->update_meta_data('_tuxopay_payment_session_id', $payment_result['payment_session_id']);
        }

        if (isset($payment_result['payment_id'])) {
            $order->update_meta_data('_tuxopay_payment_id', $payment_result['payment_id']);
        }

        if (isset($payment_result['provider'])) {
            $order->update_meta_data('_tuxopay_provider', $payment_result['provider']);
        }

        $order->update_status('pending', __('Awaiting TuxoPay payment.', 'tuxopay-orchestrator'));
        $order->save();

        // Redirect to TuxoPay hosted payment page
        return array(
            'result'   => 'success',
            'redirect' => $payment_result['redirect_url'],
        );
    }

    /**
     * Initiate payment flow via TuxoPay API
     */
    private function initiate_payment_flow($order) {
        $endpoint = $this->api_url . '/api/payment-flows/initiate';

        // Build order items
        $items = array();
        foreach ($order->get_items() as $item) {
            $items[] = array(
                'name' => $item->get_name(),
                'price' => floatval($item->get_total()) / $item->get_quantity(), // Unit price
                'quantity' => $item->get_quantity(),
            );
        }

        $body = array(
            'flow_name' => $this->flow_name,
            'amount' => $order->get_total(),
            'currency' => $order->get_currency(),
            'description' => sprintf(__('Order #%s', 'tuxopay-orchestrator'), $order->get_order_number()),
            'customer' => array(
                'email' => $order->get_billing_email(),
                'first_name' => $order->get_billing_first_name(),
                'last_name' => $order->get_billing_last_name(),
                'phone' => $order->get_billing_phone() ?: null,
                'address' => $order->get_billing_address_1() ?: null,
                'city' => $order->get_billing_city() ?: null,
                'region' => $order->get_billing_state() ?: null,
                'postal_code' => $order->get_billing_postcode() ?: null,
                'country_code' => $order->get_billing_country() ?: null,
            ),
            'items' => $items,
            'metadata' => array(
                'order_id' => $order->get_id(),
                'order_number' => $order->get_order_number(),
            ),
            'return_url' => $this->get_return_url($order),
            'cancel_url' => wc_get_checkout_url(),
            'webhook_endpoint' => WC()->api_request_url('tuxopay_webhook'),
        );

        // Log the request
        error_log('TuxoPay API Request to: ' . $endpoint);
        error_log('TuxoPay API Request body: ' . json_encode($body));

        $response = wp_remote_post($endpoint, array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'application/json',
            ),
            'body' => json_encode($body),
            'timeout' => 70,
        ));

        if (is_wp_error($response)) {
            error_log('TuxoPay API WP_Error: ' . $response->get_error_message());
            return $response;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = json_decode(wp_remote_retrieve_body($response), true);

        // Log the response
        error_log('TuxoPay API Response Code: ' . $response_code);
        error_log('TuxoPay API Response Body: ' . wp_remote_retrieve_body($response));

        if ($response_code !== 200) {
            // Try to get error from different possible locations in response
            $error_message = 'Unknown error';
            if (isset($response_body['error'])) {
                $error_message = $response_body['error'];
            } elseif (isset($response_body['data']['error'])) {
                $error_message = $response_body['data']['error'];
            } elseif (isset($response_body['message'])) {
                $error_message = $response_body['message'];
            }

            error_log('TuxoPay API Error: ' . $error_message);
            return new WP_Error('api_error', $error_message);
        }

        return $response_body;
    }
}
